<?php
/* --------------------------------------------------------------
   TrackingCodeReader.php 2020-03-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\TrackingCode\Repository;

use Doctrine\DBAL\Connection;
use Gambio\Admin\TrackingCode\Exceptions\TrackingCodeNotFoundException;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeId;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeSqlCriteria;
use Gambio\Admin\TrackingCode\Interfaces\TrackingCodeSqlPagination;

/**
 * Class TrackingCodeReader
 *
 * @package Gambio\Admin\TrackingCode\Repository
 */
class TrackingCodeReader
{
    /**
     * @var Connection
     */
    private $db;
    
    
    /**
     * TrackingCodeReader constructor.
     *
     * @param Connection $db
     */
    public function __construct(Connection $db)
    {
        $this->db = $db;
    }
    
    
    /**
     * @param TrackingCodeSqlCriteria   $criteria
     * @param TrackingCodeSqlPagination $pagination
     *
     * @return array
     */
    public function getAllData(
        TrackingCodeSqlCriteria $criteria,
        TrackingCodeSqlPagination $pagination
    ): array {
        $query = $this->db->createQueryBuilder()->select('*')->from('orders_parcel_tracking_codes');
        $criteria->applyToQuery($query);
        $pagination->applyToQuery($query);
        
        return $query->execute()->fetchAll();
    }
    
    
    /**
     * @param TrackingCodeSqlCriteria $criteria
     *
     * @return int
     */
    public function getTotalCount(TrackingCodeSqlCriteria $criteria): int
    {
        $query = $this->db->createQueryBuilder()
            ->select('orders_parcel_tracking_code_id')
            ->from('orders_parcel_tracking_codes');
        $criteria->applyToQuery($query);
        
        return $query->execute()->rowCount();
    }
    
    
    /**
     * @param TrackingCodeId $id
     *
     * @return array
     *
     * @throws TrackingCodeNotFoundException
     */
    public function getById(TrackingCodeId $id): array
    {
        $trackingCodeData = $this->db->createQueryBuilder()
            ->select('*')
            ->from('orders_parcel_tracking_codes')
            ->where('orders_parcel_tracking_code_id = :trackingCodeId')
            ->setParameter('trackingCodeId', $id->value())
            ->execute()
            ->fetch();
        
        if ($trackingCodeData === false) {
            throw TrackingCodeNotFoundException::forId($id->value());
        }
        
        return $trackingCodeData;
    }
}